<?php
namespace BooklyCustomerCabinet\Frontend\Modules\CustomerCabinet;

use BooklyCustomerCabinet\Lib;
use Bookly\Lib as BooklyLib;

/**
 * Class Controller
 * @package BooklyCustomerCabinet\Frontend\Modules\CustomerCabinet
 */
class ShortCode extends BooklyLib\Base\Component
{
    /**
     * Init component.
     */
    public static function init()
    {
        // Register shortcodes.
        add_shortcode( 'bookly-customer-cabinet', array( __CLASS__, 'render' ) );

        // Assets.
        add_action( 'wp_enqueue_scripts', array( __CLASS__, 'linkStyles' ) );
        add_action( 'wp_enqueue_scripts', array( __CLASS__, 'linkScripts' ) );
    }

    /**
     * Link styles.
     */
    public static function linkStyles()
    {
        if (
            get_option( 'bookly_gen_link_assets_method' ) == 'enqueue' ||
            BooklyLib\Utils\Common::postsHaveShortCode( 'bookly-customer-cabinet' )
        ) {
            $bookly_ver = BooklyLib\Plugin::getVersion();
            $bookly_url = plugins_url( '', BooklyLib\Plugin::getMainFile() );
            $cabinet    = plugins_url( '', Lib\Plugin::getMainFile() );

            wp_enqueue_style( 'bookly-ladda.min.css', $bookly_url . '/frontend/resources/css/ladda.min.css', array(), $bookly_ver );
            if ( get_option( 'bookly_cst_phone_default_country' ) != 'disabled' ) {
                wp_enqueue_style( 'bookly-intlTelInput.css', $bookly_url . '/frontend/resources/css/intlTelInput.css', array(), $bookly_ver );
            }
            wp_enqueue_style( 'bookly-fontawesome-all.min.css', $bookly_url . '/backend/resources/css/fontawesome-all.min.css', array(), $bookly_ver );
            wp_enqueue_style( 'bookly-bootstrap.min.css', $bookly_url . '/backend/resources/bootstrap/css/bootstrap.min.css', array(), $bookly_ver );
            wp_enqueue_style( 'bookly-customer-cabinet.css', $cabinet . '/frontend/modules/customer_cabinet/resources/css/customer-cabinet.css', array( 'bookly-bootstrap.min.css' ), Lib\Plugin::getVersion() );
        }
    }

    /**
     * Link scripts.
     */
    public static function linkScripts()
    {
        if (
            get_option( 'bookly_gen_link_assets_method' ) == 'enqueue' ||
            BooklyLib\Utils\Common::postsHaveShortCode( 'bookly-customer-cabinet' )
        ) {
            $bookly_ver = BooklyLib\Plugin::getVersion();
            $bookly_url = plugins_url( '', BooklyLib\Plugin::getMainFile() );
            $cabinet    = plugins_url( '', Lib\Plugin::getMainFile() );

            wp_enqueue_script( 'bookly-spin.min.js', $bookly_url . '/frontend/resources/js/spin.min.js', array(), $bookly_ver );
            wp_enqueue_script( 'bookly-ladda.min.js', $bookly_url . '/frontend/resources/js/ladda.min.js', array( 'bookly-spin.min.js' ), $bookly_ver );
            if ( get_option( 'bookly_cst_phone_default_country' ) != 'disabled' ) {
                wp_enqueue_script( 'bookly-intlTelInput.min.js', $bookly_url . '/frontend/resources/js/intlTelInput.min.js', array( 'jquery' ), $bookly_ver );
            }
            wp_enqueue_script( 'bookly-angular.min.js', $bookly_url . '/backend/resources/js/angular.min.js', array( 'jquery' ), $bookly_ver );
            wp_enqueue_script( 'bookly-moment.min.js', $bookly_url . '/backend/resources/js/moment.min.js', array(), $bookly_ver );
            wp_enqueue_script( 'bookly-daterangepicker.js', $bookly_url . '/backend/resources/js/daterangepicker.js', array( 'jquery', 'bookly-moment.min.js' ), $bookly_ver );
            wp_enqueue_script( 'bookly-bootstrap.min.js', $bookly_url . '/backend/resources/bootstrap/js/bootstrap.min.js', array( 'jquery' ), $bookly_ver );
            wp_enqueue_script( 'bookly-alert.js', $bookly_url . '/backend/resources/js/alert.js', array(), $bookly_ver );
            wp_enqueue_script( 'bookly-datatables.min.js', $bookly_url . '/backend/resources/js/datatables.min.js', array( 'jquery' ), $bookly_ver );
            wp_enqueue_script( 'bookly-customer-cabinet.js', $cabinet . '/frontend/modules/customer_cabinet/resources/js/customer-cabinet.js', array( 'jquery', 'bookly-daterangepicker.js' ), Lib\Plugin::getVersion() );

            wp_localize_script( 'bookly-customer-cabinet.js', 'BooklyCustomerCabinetL10n', array(
                'csrf_token'              => BooklyLib\Utils\Common::getCsrfToken(),
                'zeroRecords'             => __( 'No appointments.', 'bookly' ),
                'minDate'                 => 0,
                'maxDate'                 => BooklyLib\Config::getMaximumAvailableDaysForBooking(),
                'datePicker'              => BooklyLib\Utils\DateTime::datePickerOptions(),
                'expired_appointment'     => __( 'Expired', 'bookly' ),
                'deny_cancel_appointment' => __( 'Not allowed', 'bookly' ),
                'cancel'                  => __( 'Cancel', 'bookly' ),
                'payment'                 => __( 'Payment', 'bookly' ),
                'reschedule'              => __( 'Reschedule', 'bookly' ),
                'noTimeslots'             => __( 'There are no time slots for selected date.', 'bookly' ),
                'profile_update_success'  => __( 'Profile updated successfully.', 'bookly' ),
                'processing'              => __( 'Processing...', 'bookly' ),
                'errors'                  => array(
                    'cancel' => __( 'Unfortunately, you\'re not able to cancel the appointment because the required time limit prior to canceling has expired.', 'bookly' ),
                    'reschedule' => __( 'The selected time is not available anymore. Please, choose another time slot.', 'bookly' )
                )
            ) );
        }
    }

    /**
     * Render Customer Services shortcode.
     *
     * @param array $attributes
     * @return string
     */
    public static function render( $attributes )
    {
        global $sitepress;

        // Disable caching.
        BooklyLib\Utils\Common::noCache();

        // Prepare URL for AJAX requests.
        $ajax_url = admin_url( 'admin-ajax.php' );

        // Support WPML.
        if ( $sitepress instanceof \SitePress ) {
            $ajax_url = add_query_arg( array( 'lang' => $sitepress->get_current_language() ) , $ajax_url );
        }

        $customer = new BooklyLib\Entities\Customer();
        if ( is_user_logged_in() && $customer->loadBy( array( 'wp_user_id' => get_current_user_id() ) ) ) {
            $titles = array(
                'category'   => BooklyLib\Utils\Common::getTranslatedOption( 'bookly_l10n_label_category' ),
                'service'    => BooklyLib\Utils\Common::getTranslatedOption( 'bookly_l10n_label_service' ),
                'staff'      => BooklyLib\Utils\Common::getTranslatedOption( 'bookly_l10n_label_employee' ),
                'location'   => BooklyLib\Utils\Common::getTranslatedOption( 'bookly_l10n_label_location' ),
                'date'       => __( 'Date', 'bookly' ),
                'time'       => __( 'Time', 'bookly' ),
                'price'      => __( 'Price', 'bookly' ),
                'online_meeting' => __( 'Online meeting', 'bookly' ),
                'join_online_meeting' => __( 'Join online meeting', 'bookly' ),
                'cancel'     => __( 'Cancel', 'bookly' ),
                'reschedule' => __( 'Reschedule', 'bookly' ),
                'status'     => __( 'Status', 'bookly' ),
            );
            if ( BooklyLib\Config::customFieldsActive() && get_option( 'bookly_custom_fields_enabled' ) ) {
                foreach ( (array) BooklyLib\Proxy\CustomFields::getTranslated() as $field ) {
                    if ( ! in_array( $field->type, array( 'captcha', 'text-content', 'file' ) ) ) {
                        $titles[ 'custom_field_' . $field->id ] = $field->label;
                    }
                }
            }

            $customer_address = array(
                'country'            => $customer->getCountry(),
                'state'              => $customer->getState(),
                'postcode'           => $customer->getPostcode(),
                'city'               => $customer->getCity(),
                'street'             => $customer->getStreet(),
                'street_number'      => $customer->getStreetNumber(),
                'additional_address' => $customer->getAdditionalAddress(),
            );

            $appointment_columns = isset( $attributes['appointments'] ) ? explode( ',', $attributes['appointments'] ) : array();
            foreach ( $appointment_columns as $pos => $column ) {
                if ( ! array_key_exists( $column, $titles ) ) {
                    unset( $appointment_columns[ $pos ] );
                }
            }

            return self::renderTemplate( 'short_code', array(
                'ajax_url'            => $ajax_url,
                'appointment_columns' => $appointment_columns,
                'attributes'          => (array) $attributes,
                'customer'            => $customer,
                'customer_address'    => $customer_address,
                'form_id'             => uniqid( 'bookly-js-customer-cabinet-' ),
                'titles'              => $titles,
            ), false );
        }

        return self::renderTemplate( 'permission', array(), false );
    }
}