<?php
/*
Plugin Name: WPUM - User Verification
Plugin URI:  https://wpusermanager.com/addons/user-verification/
Description: Send a verification link via email upon registration or manually approve users before they can login. This is an addon for WP User Manager.
Version: 1.0.2
Author:      Alessandro Tesoro
Author URI:  http://wpusermanager.com
License:     GPLv2+
Text Domain: wpum-user-verification
Domain Path: /languages
*/

/**
 * Copyright (c) 2016 Alessandro Tesoro
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2 or, at
 * your discretion, any later version, as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WPUM_User_Verification' ) ) :

/**
 * Main WPUM_User_Verification Class
 *
 * @since 1.0.0
 */
class WPUM_User_Verification {

	/**
   * Instance of the addon.
   *
   * @since 1.0.0
   * @var instance of the addon.
   */
	private static $instance;

	/**
   * Get activated instance.
   *
   * @return object instance
   * @since 1.0.0
   */
	public static function instance() {
	    if( ! self::$instance ) {

					self::$instance = new WPUM_User_Verification();
	        self::$instance->setup_constants();
					self::$instance->includes();

					add_action( 'plugins_loaded', array( self::$instance, 'load_textdomain' ) );

	    }
	    return self::$instance;
	}

	/**
	 * Throw error on object clone
	 *
	 * The whole idea of the singleton design pattern is that there is a single
	 * object therefore, we don't want the object to be cloned.
	 *
	 * @access protected
	 * @return void
	 * @since 1.0.0
	 */
	public function __clone() {
		// Cloning instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'wpum-user-verification' ), '1.0.0' );
	}

	/**
	 * Disable unserializing of the class
	 *
	 * @access protected
	 * @return void
	 * @since 1.0.0
	 */
	public function __wakeup() {
		// Unserializing instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'wpum-user-verification' ), '1.0.0' );
	}

	/**
	 * Setup plugin constants.
	 *
	 * @access private
	 * @return void
	 * @since 1.0.0
	 */
	private function setup_constants() {

		// Plugin version
		if ( ! defined( 'WPUMUV_VERSION' ) ) {
			define( 'WPUMUV_VERSION', '1.0.2' );
		}

		// Plugin Folder Path
		if ( ! defined( 'WPUMUV_PLUGIN_DIR' ) ) {
			define( 'WPUMUV_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
		}

		// Plugin Folder URL
		if ( ! defined( 'WPUMUV_PLUGIN_URL' ) ) {
			define( 'WPUMUV_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
		}

		// Plugin Root File
		if ( ! defined( 'WPUMUV_PLUGIN_FILE' ) ) {
			define( 'WPUMUV_PLUGIN_FILE', __FILE__ );
		}

		// Plugin Slug
		if ( ! defined( 'WPUMUV_SLUG' ) ) {
			define( 'WPUMUV_SLUG', plugin_basename( __FILE__ ) );
		}

		define( 'WPUMUV_PRODUCT_NAME', 'User Verification' );

	}

	/**
	 * Include required files.
	 *
	 * @access private
	 * @return void
	 * @since 1.0.0
	 */
	private function includes() {

		//require_once WPUMUV_PLUGIN_DIR . 'includes/install.php';
		require_once WPUMUV_PLUGIN_DIR . 'includes/functions.php';
		require_once WPUMUV_PLUGIN_DIR . 'includes/hooks.php';
		require_once WPUMUV_PLUGIN_DIR . 'includes/class-wpumuv-login-lock.php';

		if( is_admin() ) {
			require_once WPUMUV_PLUGIN_DIR . 'includes/class-wpumuv-approval-email.php';
			require_once WPUMUV_PLUGIN_DIR . 'includes/class-wpumuv-reject-email.php';
			require_once WPUMUV_PLUGIN_DIR . 'includes/class-wpumuv-verification-email.php';
			require_once WPUMUV_PLUGIN_DIR . 'includes/class-wpumuv-pending-users-table.php';
			require_once WPUMUV_PLUGIN_DIR . 'includes/class-wpumuv-signups-manager.php';
			require_once WPUMUV_PLUGIN_DIR . 'includes/options.php';
		}

		if( class_exists( 'WPUM_License' ) ) {

			$wpumuv_license = new WPUM_License(
				__FILE__,
				WPUMUV_PRODUCT_NAME,
				WPUMUV_VERSION,
				'Alessandro Tesoro'
			);

		}

	}

	/**
	 * Load the language files for translation.
	 *
	 * @access public
	 * @return void
	 * @since 1.0.0
	 */
	public function load_textdomain() {

		// Set filter for plugin's languages directory
		$wpum_lang_dir = dirname( plugin_basename( WPUMUV_PLUGIN_FILE ) ) . '/languages/';
		$wpum_lang_dir = apply_filters( 'wpumuv_languages_directory', $wpum_lang_dir );

		// Traditional WordPress plugin locale filter
		$locale        = apply_filters( 'plugin_locale',  get_locale(), 'wpum-user-verification' );
		$mofile        = sprintf( '%1$s-%2$s.mo', 'wpum-user-verification', $locale );

		// Setup paths to current locale file
		$mofile_local  = $wpum_lang_dir . $mofile;
		$mofile_global = WP_LANG_DIR . '/wpum-user-verification/' . $mofile;

		if ( file_exists( $mofile_global ) ) {
			// Look in global /wp-content/languages/wpum-user-verification folder
			load_textdomain( 'wpum-user-verification', $mofile_global );
		} elseif ( file_exists( $mofile_local ) ) {
			// Look in local /wp-content/plugins/wpum-user-verification/languages/ folder
			load_textdomain( 'wpum-user-verification', $mofile_local );
		} else {
			// Load the default language files
			load_plugin_textdomain( 'wpum-user-verification', false, $wpum_lang_dir );
		}

	}

	/**
	 * Display version mismatch message if base plugin does not meet minimum requirement.
	 *
	 * @access public
	 * @return void
	 * @since 1.0.0
	 */
	public static function version_mismatch() {
		?>
		<div class="error">
			<p><?php printf( esc_html__( 'The WPUM - User Verification Addon requires version 1.3.0 or greater of the WP User Manager plugin. Please %s to continue.', 'wpum-user-verification' ), '<a href="'.admin_url( 'update-core.php' ).'">update it</a>'  ); ?></p>
		</div>
		<?php
	}

}

endif;

/**
 * Handles activation of the addon.
 *
 * @return mixed
 * @since 1.0.0
 */
function WPUM_UserVerification_Load() {

	if( ! class_exists( 'WP_User_Manager' ) ) {

		// Show message if plugin is not activated.
		if( ! class_exists( 'WPUM_Extension_Activation' ) ) {
			require_once 'includes/class.extension-activation.php';
		}
		$activation = new WPUM_Extension_Activation( plugin_dir_path( __FILE__ ), basename( __FILE__ ) );
		$activation = $activation->run();

	} else {

		$base_version     = defined( 'WPUM_VERSION' ) ? WPUM_VERSION : false;
		$required_version = '1.3.0';

		if ( version_compare( $base_version, $required_version, '<' ) ) {
			return add_action( 'admin_notices', 'WPUM_User_Verification::version_mismatch' );
		}

		return WPUM_User_Verification::instance();

	}

}
add_action( 'plugins_loaded', 'WPUM_UserVerification_Load' );

/**
 * Installation hook.
 *
 * @since 1.0.0
 * @return void
 */
function wpumuv_install() {

	// Setup cron jobs.
  wp_clear_scheduled_hook( 'wpumuv_check_non_verified_users' );
  wp_schedule_event( time(), 'daily', 'wpumuv_check_non_verified_users' );

	// Install new emails.
	$emails = get_option( 'wpum_emails', array() );

	if( ! wpum_email_exists( 'approval' ) ) {

		$subject = sprintf( esc_html__( 'Your account at %s has been approved.', 'wpum-user-verification' ), get_option( 'blogname' ) );
		$message = __( "Hello {username}, \n\n", 'wpum-user-verification' );
		$message .= __( "Congratulations, your account has now been approved. \n\n", 'wpum-user-verification' );

		$emails[ 'approval' ] = array(
			'subject' => $subject,
			'message' => $message
		);

	}

	if( ! wpum_email_exists( 'reject' ) ) {

		$subject = sprintf( esc_html__( 'Your registration at %s has been rejected.', 'wpum-user-verification' ), get_option( 'blogname' ) );
		$message = __( "Hello {username}, \n\n", 'wpum-user-verification' );
		$message .= __( "Unfortunately, your registration has been rejected and your account deleted. \n\n", 'wpum-user-verification' );

		$emails[ 'reject' ] = array(
			'subject' => $subject,
			'message' => $message
		);

	}

	if( ! wpum_email_exists( 'verification' ) ) {

		$subject = sprintf( esc_html__( 'Verify your account at %s', 'wpum-user-verification' ), get_option( 'blogname' ) );
		$message = __( "Hello {username}, \n\n", 'wpum-user-verification' );
		$message .= __( "Please click the link below to verify your account \n\n", 'wpum-user-verification' );
    $message .= __( "{verification_link} \n\n", 'wpum-user-verification' );

		$emails[ 'verification' ] = array(
			'subject' => $subject,
			'message' => $message
		);

	}

}
register_activation_hook( __FILE__, 'wpumuv_install' );
