<?php
namespace Arkfield\Admin\ThemeCustomizer;

if ( ! class_exists( __NAMESPACE__ . '\\Theme_Options_Config' ) ) {
	/**
	 * Defines the configuration for the theme customizer
	 *
	 * Class Theme_Options_Config
	 * @package Arkfield\Admin\ThemeCustomizer
	 */
	class Theme_Options_Config {
		const PERSIST_SETTING_DEFAULT = true;
		const DEFAULT_CONTROL_TRANSPORT_TYPE = 'postMessage';
		const FALLBACK_FONT_FAMILY = 'PT Sans, sans-serif';

		const CONTROL_TYPE_TEXT = 'text';
		const CONTROL_TYPE_NUMBER = 'number';
		const CONTROL_TYPE_RANGE = 'range';
		const CONTROL_TYPE_SELECT = 'select';
		const CONTROL_TYPE_IMAGE = 'image';
		const CONTROL_TYPE_COLOR = 'color';
		const CONTROL_TYPE_CHECKBOX = 'checkbox';
		const CONTROL_TYPE_TEXTAREA = 'textarea';

		const DEFAULT_CONTROL_TYPE = self::CONTROL_TYPE_TEXT;

		/**
		 * @var Google_Fonts_Options
		 */
		protected $google_fonts_options;

		/**
		 * Theme_Options_Config constructor.
		 *
		 * @param Google_Fonts_Options $google_fonts_options
		 */
		public function __construct( Google_Fonts_Options $google_fonts_options ) {
			$this->google_fonts_options = $google_fonts_options;
		}


		/**
		 * @return array
		 */
		public function get_customizations() {
			return array(
				'panels'   => array(
					array(
						'id'          => 'general',
						'title'       => esc_html__( 'THEME OPTIONS', 'arkfield' ),
						'description' => esc_html__( 'General theme panel', 'arkfield' ),
						'sections'    => array(
							array(
								'id'       => 'header',
								'title'    => esc_html__( 'Header', 'arkfield' ),
								'controls' => $this->get_header_section_controls(),
							),
							array(
								'id'       => 'footer',
								'title'    => esc_html__( 'Footer', 'arkfield' ),
								'controls' => $this->get_footer_section_control(),
							),
							array(
								'id'       => 'typography',
								'title'    => esc_html__( 'Typography', 'arkfield' ),
								'controls' => $this->get_cartography_section_controls(),
							),
							array(
								'id'       => 'diary',
								'title'    => esc_html__( 'Diary', 'arkfield' ),
								'controls' => $this->get_diary_section_controls(),
							),
							array(
								'id'       => 'projects',
								'title'    => esc_html__( 'Projects', 'arkfield' ),
								'controls' => $this->get_projects_section_controls(),
							),
							array(
								'id'       => 'custom_css',
								'title'    => esc_html__( 'Custom CSS', 'arkfield' ),
								'controls' => $this->get_custom_css_section_controls(),
							),
						),
					),
				),
				'sections' => array(
					array(
						'id'       => 'colors',
						'built-in' => true,
						'controls' => $this->get_colors_section_controls(),
					),
				),
			);
		}

		/**
		 * @return array
		 */
		public function get_controls() {
			$customizations = $this->get_customizations();
			$sections       = $customizations['sections'];
			$controls       = array();

			foreach ( $customizations['panels'] as $panel ) {
				$sections = array_merge( $sections, $panel['sections'] );
			}

			foreach ( $sections as $section ) {
				foreach ( $section['controls'] as $control ) {
					$controls[ $control['id'] ] = $control;
				}
			}

			return $controls;
		}

		/**
		 * @return array
		 */
		public function get_header_section_controls() {
			return array(
				array(
					'id'          => 'logo_image',
					'type'        => self::CONTROL_TYPE_IMAGE,
					'label'       => esc_html__( 'Upload your logo', 'arkfield' ),
					'description' => esc_html__( '_the dimensions of your logo image are totally up to you', 'arkfield' ),
				),
				array(
					'id'          => 'logo_dimension',
					'type'        => self::CONTROL_TYPE_RANGE,
					'default'     => 50,
					'label'       => esc_html__( 'Logo dimension', 'arkfield' ),
					'description' => esc_html__( '_scale your logo image to best fit your site header', 'arkfield' ),
					'input_attrs' => array(
						// do not change this value, because it is only used for scaling, manipulate only data fields
						'min'             => 0,
						'max'             => 100,
						'step'            => 1,
						'data-min-width'  => 10,
						'data-max-width'  => 180,
						'data-min-height' => 10,
						'data-max-height' => 96,
					),
				),
				array(
					'id'          => 'logo_text',
					'type'        => self::CONTROL_TYPE_TEXT,
					'default'     => 'arkfield',
					'label'       => esc_html__( 'Logo text', 'arkfield' ),
					'description' => esc_html__( '_it will appear in case you do not have an image for your logo', 'arkfield' ),
					'css_items'   => array(
						'#logo-text',
					),
				),
			);
		}

		/**
		 * @return array
		 */
		public function get_diary_section_controls() {
			return array(
				array(
					'id'          => 'blog_title',
					'type'        => self::CONTROL_TYPE_TEXT,
					'default'     => esc_html__( 'Follow our diary', 'arkfield' ),
					'label'       => esc_html__( 'Diary title', 'arkfield' ),
					'description' => esc_html__( '_insert here the title of your Blog Page', 'arkfield' ),
					'css_items'   => array(
						'.main__header h1',
					),
				),
				array(
					'id'          => 'posts_speed',
					'type'        => self::CONTROL_TYPE_CHECKBOX,
					'default'     => 1,
					'label'       => esc_html__( 'Posts Speed', 'arkfield' ),
					'description' => esc_html__( '_check this if you want your posts moving faster on Posts Page', 'arkfield' ),
					'transport'   => 'refresh',
				)
			);
		}

		/**
		 * @return array
		 */
		public function get_footer_section_control() {
			return array(
				array(
					'id'          => 'footer_widgets_columns',
					'type'        => self::CONTROL_TYPE_SELECT,
					'default'     => 3,
					'label'       => esc_html__( 'Widgets', 'arkfield' ),
					'description' => esc_html__( '_choose the number of columns your widgets will best fit in the footer', 'arkfield' ),
					'choices'     => array(
						1 => 1,
						2 => 2,
						3 => 3,
						4 => 4,
					),
					'css'         => array(
						'width' => array( '.widget-item' ),
					),
					'css_filter'  => 'reverse_percentage',
				),
				array(
					'id'        => 'copyright_text',
					'type'      => self::CONTROL_TYPE_TEXT,
					'default'   => sprintf( 'Copyright &copy; %s %s', PIKARTHOUSE_THEME_VENDOR, date( 'Y' ) ),
					'label'     => esc_html__( 'Copyright text', 'arkfield' ),
					'css_items' => array(
						'.copyright-area',
					),
				),
			);
		}

		/**
		 * @return array
		 */
		public function get_cartography_section_controls() {
			return array(
				array(
					'id'         => 'heading_font_family',
					'type'       => self::CONTROL_TYPE_SELECT,
					'default'    => 'Oswald:300,400,700',
					'label'      => esc_html__( 'Heading font family', 'arkfield' ),
					'choices'    => $this->get_google_font_to_name_list(),
					'css'        => array(
						'font-family' => array(
							'h1',
							'h2',
							'h3',
							'h4',
							'h5',
							'h6',
							'.form-control',
							'#logo-text',
							'.menu-item',
							'.time-container',
							'.copyright-area',
							'.btn--top',
							'.related-posts li',
							'.quote__author-name',
							'.widget_calendar caption',
							'.widget_calendar tbody',
							'.widget_calendar thead th',
							'.widget_calendar tfoot td a',
							'.taxonomies .categories',
							'.taxonomies .tags',
							'.widget_categories',
							'.tagcloud',
							'.nav-links',
							'.navigation--page-breaks>span',
							'.navigation--page-breaks>a',
							'.page-numbers',
							'.pikode--btn',
							'.pikode--btn--reverse',
							'.mesh__filter a',
							'.wpcf7-form input[type=submit]',
							'.wc-tabs__link',
							'.woocommerce button.button.alt',
							'.woocommerce .quantity .qty',
							'.woocommerce-product-price .price',
							'.woocommerce span.onsale',
							'.woocommerce-products span.onsale',
							'.woocommerce-out-stock',
							'.product__image--bottom__cart',
							'.added_to_cart',
							'.woocommerce a.button.alt',
							'.woocommerce input.button',
							'.coupon-button',
							'.shipping-button',
							'.woocommerce .woocommerce-message .button',
							'.product-categories',
							'.widget_price_filter',
							'.widget_product_search .search-field',
							'.woocommerce a.button',
							'.orderby',
							'.select-temp',
							'.wc-cart--link__text'
						),
					),
					'css_filter' => 'font_family',
				),
				array(
					'id'          => 'heading_line_height',
					'type'        => self::CONTROL_TYPE_NUMBER,
					'default'     => 1.2,
					'label'       => esc_html__( 'Line height', 'arkfield' ),
					'input_attrs' => array(
						'min'  => 0,
						'max'  => 5,
						'step' => 0.1,
					),
					'css'         => array(
						'line-height' => array(
							'h1',
							'h2',
							'h3',
							'h4',
							'h5',
							'h6',
							'.time-container',
						),
					),
				),
				array(
					'id'         => 'text_font_family',
					'type'       => self::CONTROL_TYPE_SELECT,
					'default'    => 'Open+Sans:300,300italic,400,400italic,600,600italic,700,700italic,800,800italic',
					'label'      => esc_html__( 'Text font family', 'arkfield' ),
					'choices'    => $this->get_google_font_to_name_list(),
					'css'        => array(
						'font-family' => array(
							'body'
						),
					),
					'css_filter' => 'font_family',
				),
				array(
					'id'          => 'text_font_size',
					'type'        => self::CONTROL_TYPE_NUMBER,
					'default'     => 16,
					'label'       => esc_html__( 'Font size (px)', 'arkfield' ),
					'input_attrs' => array(
						'min'  => 6,
						'max'  => 100,
						'step' => 1,
					),
					'css'         => array(
						'font-size' => array(
							'body'
						),
					),
					'css_filter'  => 'size_unit_px',
				),
				array(
					'id'          => 'text_line_height',
					'type'        => self::CONTROL_TYPE_NUMBER,
					'default'     => 1.7,
					'label'       => esc_html__( 'Line height', 'arkfield' ),
					'input_attrs' => array(
						'min'  => 0,
						'max'  => 5,
						'step' => 0.1,
					),
					'css'         => array(
						'line-height' => array(
							'body'
						),
					),
				),
				array(
					'id'          => 'letter_spacing',
					'type'        => self::CONTROL_TYPE_NUMBER,
					'default'     => 0,
					'label'       => esc_html__( 'Letter spacing (px)', 'arkfield' ),
					'input_attrs' => array(
						'min'  => 0,
						'max'  => 10,
						'step' => 0.1,
					),
					'css'         => array(
						'letter-spacing' => array(
							'body'
						),
					),
					'css_filter'  => 'size_unit_px',
				),
				array(
					'id'          => 'google_font',
					'type'        => self::CONTROL_TYPE_TEXT,
					'label'       => esc_html__( 'Google Fonts', 'arkfield' ),
					'description' => esc_html__( '_add your custom Google Font in the field below Ex: Ubuntu+Mono:400,400italic,700', 'arkfield' ),
					'persist'     => false,
				),
				array(
					'id'      => 'add_google_font',
					'label'   => false,
					'type'    => 'button',
					'default' => esc_html__( 'Add font', 'arkfield' ),
					'persist' => false,
				)
			);
		}

		/**
		 * @return array
		 */
		public function get_colors_section_controls() {
			return array(
				array(
					'id'      => 'headings_color',
					'type'    => self::CONTROL_TYPE_COLOR,
					'default' => '#363636',
					'label'   => esc_html__( 'Headings Color', 'arkfield' ),
					'css'     => array(
						'color' => array(
							'h1',
							'h1 a',
							'h2',
							'h2 a',
							'h3',
							'h3 a',
							'h4',
							'h4 a',
							'h5',
							'h5 a',
							'h6',
							'h6 a',
						),
					),
				),
				array(
					'id'      => 'text_color',
					'type'    => self::CONTROL_TYPE_COLOR,
					'default' => '#000',
					'label'   => esc_html__( 'Text Color', 'arkfield' ),
					'css'     => array(
						'color' => array(
							'body',
						),
					),
				),
				array(
					'id'      => 'feature_color',
					'type'    => self::CONTROL_TYPE_COLOR,
					'default' => '#333540',
					'label'   => esc_html__( 'Feature Color', 'arkfield' ),
					'css'     => array(
						'background'   => array(
							// Preloader
							'#preloader',
							// Header
							'.device--mobile .flexbox__item--menus',
							'.sub-menu',
							'.menu--main>.menu-item:before',
							// Sticky
							'.sticky .time-container',
							'.sticky .read-more',
							// Pagination
							'.pagination .current',
							'a.page-numbers:hover',
							'a.page-numbers:focus',
							'span.page-numbers:hover',
							'span.page-numbers:focus',
							'.navigation--comments a',
							'.navigation--page-breaks>span',
							'.navigation--page-breaks>a:hover',
							'.navigation--page-breaks>a:focus',
							// Shortcodes
							'.pikode--btn',
							'.pikode--btn--reverse:hover',
							'.pikode--btn--reverse:focus',
							'.pikode--icon.circle',
							'.pikode--icon.rectangle',
							'.progressbar__progress',
							'.team-member__image__container:after',
							// Posts
							'.read-more:hover',
							'.read-more:focus',
							'.categories h6',
							'.tags h6',
							'.comment-reply a:hover',
							'.comment-reply a:focus',
							'.depth-1> .comment-body .comment-reply a',
							'.mejs-container .mejs-inner .mejs-controls',
							// Projects
							'.mesh__item__link:after',
							'.mesh__filter a:before',
							'.btn-selected:before',
							'.btn--selected',
							// Footer
							'#site-footer',
							// Plugins
							'.mfp-bg',
							'.rsBullet span:hover',
							'.rsBullet span:focus',
							'.rsBullet.rsNavSelected span',
							'.wpcf7-submit',
							'.woocommerce div.product .woocommerce-tabs ul.tabs li.active',
							'.wc-tabs__link:before',
							'.woocommerce button.button.alt',
							'.woocommerce span.onsale',
							'.woocommerce-products span.onsale',
							'.woocommerce-out-stock',
							'.product__image--bottom__cart:hover',
							'.product__image--bottom__cart:focus',
							'.product__image--bottom__view:hover',
							'.product__image--bottom__view:focus',
							'.added_to_cart:hover',
							'.added_to_cart:focus',
							'.woocommerce a.button.alt',
							'.coupon-button:hover',
							'.coupon-button:focus',
							'.woocommerce input.button:hover',
							'.woocommerce input.button:focus',
							'.woocommerce a.remove:hover',
							'.woocommerce a.remove:focus',
							'.shipping-button:hover',
							'.shipping-button:focus',
							'.woocommerce-cart .quantity .qty',
							'.woocommerce .woocommerce-message .button',
							'.woocommerce input.button.alt',
							'p.demo_store',
							'.wc-cart--items-number:hover',
							'.wc-cart--items-number:focus',
							'.woocommerce nav.woocommerce-pagination ul li span.current',
							'.woocommerce nav.woocommerce-pagination ul li span:hover',
							'.woocommerce nav.woocommerce-pagination ul li span:focus',
							'.woocommerce nav.woocommerce-pagination ul li a:hover',
							'.woocommerce nav.woocommerce-pagination ul li a:focus'
						),
						'color'        => array(
							'.menu--social .menu-item a:hover',
							'.menu--social .menu-item a:focus',
							'.pikode--color',
							'.product__image--bottom__rating .star-rating span'
						),
						'border-color' => array(
							'.icon-border',
						)
					),
				),
			);
		}

		protected function get_projects_section_controls() {
			return array(
				array(
					'id'          => 'projects_lazy_load',
					'type'        => self::CONTROL_TYPE_CHECKBOX,
					'default'     => 1,
					'label'       => esc_html__( 'Lazy Load', 'arkfield' ),
					'description' => esc_html__( '_check this if you want your projects to load lazily', 'arkfield' ),
					'transport'   => 'refresh',
				)
			);
		}

		/**
		 * @return array
		 */
		public function get_custom_css_section_controls() {
			return array(
				array(
					'id'          => 'custom_css',
					'type'        => self::CONTROL_TYPE_TEXTAREA,
					'label'       => esc_html__( 'Custom CSS', 'arkfield' ),
					'description' => esc_html__( '_insert your custom CSS below', 'arkfield' ),
					'transport'   => 'refresh',
				),
			);
		}

		/**
		 * @return array
		 */
		public function get_options_for_transport_type() {
			return array(
				'blogname',
				'blogdescription',
				'header_textcolor',
			);
		}

		/**
		 * @return string
		 */
		public function get_theme_customizer_js_file() {
			return ARKFIELD_THEME_JS_URI . '/admin/theme-customizer.js';
		}

		/**
		 * @return array
		 */
		protected function get_google_font_to_name_list() {
			$fonts = $this->google_fonts_options->get_google_fonts();

			$font_to_name_list = array();

			foreach ( $fonts as $font ) {
				$fontData                   = explode( ':', $font );
				$font_to_name_list[ $font ] = str_replace( '+', ' ', $fontData[0] );
			}

			return $font_to_name_list;
		}
	}

}