<?php
namespace Arkfield\Setup;

use Arkfield\Admin\ThemeCustomizer\Theme_Options_Utils;

if ( ! class_exists( __NAMESPACE__ . '\Theme_Setup' ) ) {
	/**
	 * Class Theme_Setup
	 * @package Arkfield\Setup
	 */
	class Theme_Setup {
		const CONTENT_WIDTH = 1000;
		const MAX_SRC_SET_IMAGE_WIDTH = 8192;

		/**
		 * @var Theme_Options_Utils
		 */
		protected $theme_options_utils;

		/**
		 * Theme_Setup constructor.
		 *
		 * @param Theme_Options_Utils $theme_options_utils
		 */
		public function __construct( Theme_Options_Utils $theme_options_utils ) {
			$this->theme_options_utils = $theme_options_utils;
		}

		public function run() {
			if ( ! isset( $GLOBALS['content_width'] ) ) {
				$GLOBALS['content_width'] = self::CONTENT_WIDTH;
			}

			$this->setup();
			$this->enqueue_scripts();
			$this->enqueue_admin_scripts();
			$this->enqueue_media_element_scripts();
			$this->theme_options_utils->enqueue_fonts();
		}

		protected function setup() {
			$add_theme_support_callback = $this->add_theme_support_callback();
			add_action( 'after_setup_theme', function () use ( $add_theme_support_callback ) {
				/*
				 * Make theme available for translation.
				 * Translations can be filed in the /languages/ directory.
				 */
				\load_theme_textdomain( 'arkfield', \get_template_directory() . '/languages' );

				$add_theme_support_callback();

				\register_nav_menus( array(
					'primary' => esc_html__( 'Primary Menu', 'arkfield' ),
					'footer'  => esc_html__( 'Footer Menu', 'arkfield' ),
					'social'  => esc_html__( 'Social Links Menu', 'arkfield' ),
				) );
			} );

			\add_filter( 'max_srcset_image_width', function () {
				return Theme_Setup::MAX_SRC_SET_IMAGE_WIDTH;
			} );
		}


		protected function enqueue_scripts() {
			$enqueue_style_callback = $this->enqueue_style_callback();

			\add_action( 'wp_enqueue_scripts', function () use ( $enqueue_style_callback ) {
				$enqueue_style_callback( 'vendor-style', ARKFIELD_THEME_CSS_URI . '/vendor.css' );
				$enqueue_style_callback( 'theme-style', \get_stylesheet_uri(), array( 'vendor-style' ) );

				if ( \is_rtl() ) {
					$enqueue_style_callback( 'theme-rtl-style', ARKFIELD_TEMPLATE_DIR_URI . '/rtl.css', array( 'theme-style' ) );
				}

				\wp_register_script(
					'arkfield' . '-gmap-api', "http://maps.google.com/maps/api/js?sensor=false&amp;language=en",
					array(),
					false,
					true
				);

				if ( ! \is_admin() ) {
					// deregister wp jquery and register it again in order to load it in footer
					\wp_deregister_script( 'jquery' );
					\wp_enqueue_script( 'jquery', '/wp-includes/js/jquery/jquery.js', array(), ARKFIELD_THEME_VERSION, true );
				}

				\wp_enqueue_script( 'arkfield' . '-vendor-script', ARKFIELD_THEME_JS_URI . '/vendor.js', array( 'jquery' ), ARKFIELD_THEME_VERSION, true );
				\wp_enqueue_script(
					'arkfield' . '-theme-script', ARKFIELD_THEME_JS_URI . '/custom.js',
					array( 'arkfield' . '-vendor-script' ), ARKFIELD_THEME_VERSION, true
				);

				if ( \is_singular() && \comments_open() && \get_option( 'thread_comments' ) ) {
					\wp_enqueue_script( 'comment-reply' );
				}
			} );

			$theme_options_utils = $this->theme_options_utils;

			\add_action( 'wp_enqueue_scripts', function () use ( $theme_options_utils ) {
				$custom_css = trim( $theme_options_utils->get_custom_option( 'custom_css' ) );

				if ( ! empty( $custom_css ) ) {
					\wp_add_inline_style( 'arkfield' . '-theme-style', $custom_css );
				}
			}, 12 );

		}

		protected function enqueue_media_element_scripts() {
			\add_action( 'wp_enqueue_scripts', function () {

				\wp_enqueue_style( 'wp-mediaelement' );
				\wp_enqueue_script( 'mediaelement' );

				\wp_deregister_script( 'wp-mediaelement' );
				\wp_register_script( 'wp-mediaelement', ARKFIELD_THEME_JS_URI . "/wp-mediaelement.js",
					array( 'mediaelement' ), ARKFIELD_THEME_VERSION, true );
			});
		}

		protected function enqueue_admin_scripts() {
			$enqueue_style_callback = $this->enqueue_style_callback();
			\add_action( 'admin_enqueue_scripts', function () use ( $enqueue_style_callback ) {
				$enqueue_style_callback( 'admin-style', ARKFIELD_THEME_CSS_URI . '/admin.css' );

				if ( \is_rtl() ) {
					$enqueue_style_callback( 'admin-rtl-style', ARKFIELD_THEME_CSS_URI . '/admin-rtl.css', array( 'admin-style' ) );
				}
			} );
		}

		/**
		 * @param string $id
		 * @param string $url
		 * @param array  $dependencies
		 * @param string $version
		 */
		protected function enqueue_style( $id, $url, $dependencies = array(), $version = ARKFIELD_THEME_VERSION ) {
			$enqueue_style_callback = $this->enqueue_style_callback();
			$enqueue_style_callback( $id, $url, $dependencies, $version );
		}

		/**
		 * @return \Closure
		 */
		protected function enqueue_style_callback() {
			return function ( $id, $url, $dependencies = array(), $version = ARKFIELD_THEME_VERSION ) {
				$dependencies = array_map( function ( $dep ) {
					return 'arkfield' . '-' . $dep;
				}, $dependencies );

				\wp_enqueue_style(
					'arkfield' . '-' . $id,
					$url,
					$dependencies,
					$version
				);
			};
		}

		protected function add_theme_support() {
			$add_theme_support_callback = $this->add_theme_support_callback();
			$add_theme_support_callback();
		}

		protected function add_theme_support_callback() {
			return function () {
				// Add default posts and comments RSS feed links to head.
				\add_theme_support( 'automatic-feed-links' );
				\add_theme_support( 'post-thumbnails' );

				\add_theme_support( 'title-tag' );

				\add_theme_support(
					'post-formats',
					array( 'gallery', 'image', 'quote', 'video', 'audio' )
				);

				\add_theme_support(
					'html5',
					array( 'search-form', 'comment-form', 'comment-list', 'gallery', 'caption', 'widgets' )
				);

				//\add_theme_support( 'custom-header', [ ] );
				//\add_theme_support( 'custom-background', [ ] );
			};
		}
	}
}