<?php
namespace Arkfield\Admin\ThemeCustomizer;

if ( ! class_exists( __NAMESPACE__ . '\\Theme_Options_Wrapper' ) ) {
	/**
	 * Class Theme_Options_Wrapper
	 * @package Arkfield\Admin\ThemeCustomizer
	 */
	class Theme_Options_Wrapper {

		/**
		 * @var int
		 */
		protected static $panel_priority = 1;

		/**
		 * @var int
		 */
		protected static $section_priority = 1;

		/**
		 * @var int
		 */
		protected static $control_priority = 1;

		/**
		 * @var \WP_Customize_Manager
		 */
		protected $wp_customize_manager;

		/**
		 * @var Theme_Options_Sanitizer
		 */
		protected $theme_options_sanitizer;

		/**
		 * @var Theme_Options_Utils
		 */
		protected $theme_options_utils;

		/**
		 * @param Theme_Options_Sanitizer $theme_options_sanitizer
		 * @param Theme_Options_Utils     $theme_options_utils
		 */
		public function __construct(
			Theme_Options_Sanitizer $theme_options_sanitizer,
			Theme_Options_Utils $theme_options_utils
		) {
			$this->theme_options_sanitizer = $theme_options_sanitizer;
			$this->theme_options_utils     = $theme_options_utils;
		}

		public function set_wp_customize_manager( \WP_Customize_Manager $wp_customize_manager ) {
			$this->wp_customize_manager = $wp_customize_manager;
		}

		/**
		 * @param string $id
		 * @param string $title
		 * @param string $description
		 * @param int    $priority
		 *
		 * @return string
		 */
		public function add_panel( $id, $title, $description = '', $priority = null ) {
			$panel_id = $this->theme_options_utils->build_panel_id( $id );
			$this->wp_customize_manager->add_panel(
				$panel_id,
				array(
					'priority'    => empty( $priority ) ? static::$panel_priority ++ : $priority,
					'capability'  => 'edit_theme_options',
					'title'       => $title,
					'description' => $description,
				)
			);

			return $panel_id;
		}

		/**
		 * @param string $id
		 * @param string $panel_id
		 * @param string $title
		 * @param string $description
		 * @param int    $priority
		 *
		 * @return string
		 */
		public function add_section( $id, $panel_id, $title = '', $description = '', $priority = null ) {
			if ( empty( $title ) ) {
				$title = $this->setting_id_to_label( $id );
			}

			$section_id = $this->theme_options_utils->build_section_id( $id );

			$this->wp_customize_manager->add_section(
				$section_id,
				array(
					'title'       => $title,
					'priority'    => empty( $priority ) ? static::$section_priority ++ : $priority,
					'capability'  => 'edit_theme_options',
					'description' => $description,
					'panel'       => $panel_id,
				)
			);

			return $section_id;
		}

		/**
		 * @param string $control_id
		 * @param string $section_id
		 * @param string $type
		 * @param array  $options
		 *
		 * @return string
		 */
		public function add_control(
			$control_id, $section_id, $type = Theme_Options_Config::DEFAULT_CONTROL_TYPE, array $options = array()
		) {
			$full_control_id = $this->add_setting(
				$control_id,
				$type,
				isset( $options['default'] ) ? $options['default'] : '',
				isset( $options['transport'] ) ? $options['transport'] : Theme_Options_Config::DEFAULT_CONTROL_TRANSPORT_TYPE,
				isset( $options['persist'] ) ? $options['persist'] : Theme_Options_Config::PERSIST_SETTING_DEFAULT
			);

			if ( ! isset( $options['label'] ) || ( empty( $options['label'] ) && $options['label'] !== false ) ) {
				$options['label'] = $this->setting_id_to_label( $control_id );
			}

			if ( ! isset( $options['priority'] ) ) {
				$options['priority'] = static::$control_priority ++;
			}

			$options = array_merge( $options,
				array(
					'label'    => $options['label'],
					'section'  => $section_id,
					'settings' => $full_control_id,
				)
			);

			$control_type_method = 'add_control_' . $type;

			if ( \method_exists( $this, $control_type_method ) ) {
				return $this->$control_type_method( $full_control_id, $options );
			}

			$options['type'] = $type;

			$this->wp_customize_manager->add_control(
				$full_control_id,
				$options
			);

			return $full_control_id;
		}

		/**
		 * @param string $full_control_id
		 * @param array  $options
		 *
		 * @return string
		 */
		protected function add_control_color( $full_control_id, array $options = array() ) {
			$this->wp_customize_manager->add_control(
				new \WP_Customize_Color_Control(
					$this->wp_customize_manager,
					$full_control_id,
					$options
				)
			);

			return $full_control_id;
		}

		/**
		 * @param string $full_control_id
		 * @param array  $options
		 *
		 * @return string
		 */
		protected function add_control_image( $full_control_id, array $options = array() ) {
			$this->wp_customize_manager->add_control(
				new \WP_Customize_Image_Control(
					$this->wp_customize_manager,
					$full_control_id,
					$options
				)
			);

			return $full_control_id;
		}

		/**
		 * @param string $setting_id
		 * @param string $control_type
		 * @param string $default_value
		 * @param string $transport
		 * @param bool   $persist
		 *
		 * @return string
		 */
		protected function add_setting(
			$setting_id,
			$control_type = Theme_Options_Config::DEFAULT_CONTROL_TYPE,
			$default_value = '',
			$transport = Theme_Options_Config::DEFAULT_CONTROL_TRANSPORT_TYPE,
			$persist = Theme_Options_Config::PERSIST_SETTING_DEFAULT
		) {
			$full_setting_id = $this->theme_options_utils->build_setting_id( $setting_id );
			$config          = array(
				'default'           => $default_value,
				'transport'         => $transport,
				'type'              => 'theme_mod',
				'capability'        => 'edit_theme_options',
				'sanitize_callback' => $this->get_control_sanitize_callback( $control_type, $persist ),
			);

			$this->wp_customize_manager->add_setting(
				$full_setting_id,
				$config
			);

			return $full_setting_id;
		}

		/**
		 * @param string $control_type
		 * @param string $persist persist or not the setting in db
		 *
		 * @return array|\closure
		 */
		protected function get_control_sanitize_callback( $control_type, $persist ) {
			// if no persistence is required, the sanitize callback will always return empty value
			if ( ! $persist ) {
				return function () {
					return '';
				};
			}

			$sanitize_method = 'sanitize_' . $control_type;

			if ( ! method_exists( $this->theme_options_sanitizer, $sanitize_method ) ) {
				$sanitize_method = 'sanitize_text';
			}

			return array( $this->theme_options_sanitizer, $sanitize_method );
		}

		/**
		 * @param string $id
		 *
		 * @return string
		 */
		protected function setting_id_to_label( $id ) {
			return ucfirst( str_replace( '_', ' ', strtolower( $id ) ) );
		}
	}
}