<?php
namespace Arkfield\Project;

if ( ! class_exists( __NAMESPACE__ . '\Project_DAL' ) ) {
	/**
	 * Class Project_DAL
	 * @package Arkfield\Project
	 */
	class Project_DAL {
		/**
		 * @param \WP_Post $project
		 *
		 * @return array|bool|\WP_Error
		 */
		public function get_project_categories( \WP_Post $project ) {
			return \get_the_terms( $project, Project_Config::PROJECT_CATEGORY_SLUG );
		}

		/**
		 * Get categories for a specific project
		 *
		 * @param \WP_Post $project project object.
		 *
		 * @return array
		 */
		public function get_project_categories_name( \WP_Post $project ) {
			$categories      = $this->get_project_categories( $project );
			$categories_name = array();

			if ( is_array( $categories ) ) {
				foreach ( $categories as $category ) {
					$categories_name[ $category->term_id ] = $category->name;
				}
			}

			return $categories_name;
		}

		/**
		 * Get categories grouped by id for a specific project
		 *
		 * @param \WP_Post $project
		 *
		 * @return array
		 */
		public function get_project_categories_grouped_by_id( \WP_Post $project ) {
			$categories         = $this->get_project_categories( $project );
			$grouped_categories = array();

			if ( is_array( $categories ) ) {
				foreach ( $categories as $category ) {
					$grouped_categories[ $category->term_id ] = $category;
				}
			}

			return $grouped_categories;
		}

		/**
		 * @param \WP_Post[] $projects
		 *
		 * @return array
		 */
		public function get_unique_projects_categories( array $projects ) {
			$categories = array();

			foreach ( $projects as $project ) {
				$categories = $categories + $this->get_project_categories_grouped_by_id( $project );
			}

			ksort( $categories );

			return $categories;
		}

		/**
		 * @param int $nb_projects
		 *
		 * @return array
		 */
		public function get_projects( $nb_projects = - 1 ) {
			return \get_posts(
				array(
					'post_type'      => Project_Config::PROJECT_SLUG,
					'posts_per_page' => $nb_projects,
					'orderby'        => 'date',
					'order'          => 'DESC',
				)
			);
		}

		/**
		 * Gets the related projects of the current project
		 *
		 * It tries to find related projects by project categories
		 *
		 * @param \WP_Post $project     project object.
		 * @param int      $nb_projects number of projects to fetch.
		 *
		 * @return \WP_Post[]
		 */
		public function get_related_projects( \WP_Post $project, $nb_projects = 5 ) {
			$category_ids = $this->get_project_categories_id( $project );

			if ( empty( $category_ids ) ) {
				return array();
			}

			return \get_posts( array(
				'post_type'      => Project_Config::PROJECT_SLUG,
				'posts_per_page' => $nb_projects,
				'exclude'        => array( $project->ID ),
				'tax_query'      => array(
					array(
						'taxonomy' => Project_Config::PROJECT_CATEGORY_SLUG,
						'field'    => 'term_id',
						'terms'    => $category_ids,
					),
				)
			) );
		}

		/**
		 * @param string $project_id
		 * @param string $size
		 * @param array  $attributes
		 *
		 * @return string
		 */
		public function get_project_thumbnail( $project_id, $size, $attributes = array() ) {
			return \get_the_post_thumbnail( $project_id, $size, $attributes );
		}

		/**
		 * @param string $img_id
		 * @param string $size
		 * @param array  $attributes
		 *
		 * @return string
		 */
		public function get_project_image( $img_id, $size, $attributes = array() ) {
			return \wp_get_attachment_image( $img_id, $size, false, $attributes );
		}

		/**
		 * @param string $project_id
		 *
		 * @return array
		 */
		public function get_project_meta( $project_id ) {
			$project_meta = \get_post_meta( $project_id );

			$final_meta = array();
			foreach ( $project_meta as $key => $value ) {
				$final_meta[ str_replace( Project_Config::PROJECT_DB_PREFIX, '', $key ) ] = $value[0];
			}

			return $final_meta;
		}

		/**
		 * @param \WP_Post $project
		 *
		 * @return array
		 */
		protected function get_project_categories_id( \WP_Post $project ) {
			$categories = $this->get_project_categories( $project );
			$ids        = array();

			if ( is_array( $categories ) ) {
				foreach ( $categories as $category ) {
					$ids[] = $category->term_id;
				}
			}

			return $ids;
		}
	}
}