<?php
/**
 * Locks the login if the user isn't approved.
 *
 * @package     wpum-user-verification
 * @copyright   Copyright (c) 2016, Alessandro Tesoro
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.1.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * WPUMUV_Login_Lock class.
 *
 * @since 1.0.0
 */
class WPUMUV_Login_Lock {

  /**
   * Get things started.
   *
   * @since 1.0.0
   */
  public function init() {

    add_filter( 'wp_authenticate_user', array( $this, 'authenticate_user' ), 1 );
    add_action( 'template_redirect', array( $this, 'verify_user' ), 10 );

  }

  /**
   * Verifies if the user can login or not.
   *
   * @since 1.0.0
   * @param  object $user the user found.
   * @return object
   */
  public function authenticate_user( $user ) {

    if ( is_wp_error( $user ) ) {
        return $user;
    }

    if( ! wpumuv_is_user_approved( $user->ID ) ) {
      return new WP_Error( 'wpumuv_not_approved', $this->get_message() );
    }

    return $user;

  }

  /**
   * Retrieves the approriate error message based on the add-on settings.
   * The message for now is showed only on wp-login.php
   *
   * @since 1.0.2
   * @return string
   */
  private function get_message() {

    $message = esc_html__( 'We have sent you a verification email. Please verify your account first.', 'wpum-user-verification' );

    $method = wpumuv_get_method();

    if( $method == 'manual' ) {
      $message = esc_html__( 'Your account is currently pending for approval.', 'wpum-user-verification' );
    }

    return $message;

  }

  /**
   * Verifies the user on the frontend.
   *
   * @return void
   */
  public function verify_user() {

    if( isset( $_GET['verify_user'] ) && isset( $_GET['verification_key'] ) && is_page( wpum_get_core_page_id( 'login' ) ) ) {

      $user_id = absint( $_GET['verify_user'] );
      $key     = $_GET['verification_key'];

      $key_exists = get_user_meta( $user_id, 'wpumuv_verification_key', true );

      if( ! empty( $key_exists ) && $key_exists == $key ) {

        wpumuv_delete_verification_status( $user_id );

        $url = get_permalink( wpum_get_core_page_id( 'login' ) );
        $url = add_query_arg( array( 'verification' => true ), $url );

        wp_redirect( $url );
        exit;

      } else {

        $url = get_permalink( wpum_get_core_page_id( 'login' ) );
        $url = add_query_arg( array( 'verification' => 'failed' ), $url );

        wp_redirect( $url );
        exit;

      }

    }

  }

}

$wpumuv_login_lock = new WPUMUV_Login_Lock;
$wpumuv_login_lock->init();
