<?php
/**
 * Signups manager page.
 *
 * @package     wpum-user-verification
 * @copyright   Copyright (c) 2016, Alessandro Tesoro
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.1.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * WPUMUV_Signups_Manager class.
 *
 * @since 1.0.0
 */
class WPUMUV_Signups_Manager {

  /**
   * Get things started.
   *
   * @since 1.0.0
   */
  public function init() {

    add_action( 'admin_menu', array( $this, 'admin_menu' ), 9 );
    add_action( 'admin_init', array( $this, 'process_signups' ), 10 );
    add_action( 'admin_notices', array( $this, 'admin_notices' ), 10 );

  }

  /**
   * Add new menu page.
   *
   * @since 1.0.0
   * @return void
   */
  public function admin_menu() {

    add_users_page( __( 'Manage Signups', 'wpum-user-verification' ), __( 'Manage Signups', 'wpum-user-verification' ), 'manage_options', 'wpum-manage-signups', array( $this, 'admin_page' ) );

  }

  /**
   * Retrieves a list of pending users.
   *
   * @since 1.0.0
   * @return array
   */
  public function get_pending_users() {

    $users = array();

    if( ! is_admin() )
      return array();

    $user_query = new WP_User_Query( array(
      'number'     => 99999,
      'meta_key'   => 'wpumuv_needs_verification',
      'meta_value' => true,
      'fields'     => array( 'ID', 'user_login', 'user_email', 'user_registered' )
    ) );

    $users = $user_query->get_results();

    return $users;

  }

  /**
   * Retrieve the user approval link.
   *
   * @since 1.0.0
   * @param  string $user_id user id number.
   * @return string
   */
  public function get_user_approve_link( $user_id ) {

    $output = wp_nonce_url( add_query_arg( array(
      'wpumuv_action' => 'approve',
      'user_id'       => $user_id
    ), admin_url( 'users.php?page=wpum-manage-signups' ) ), 'wpumuv_action' );

    return $output;

  }

  /**
   * Retrieve the user rejection link.
   *
   * @since 1.0.0
   * @param  string $user_id user id number.
   * @return string
   */
  public function get_user_reject_link( $user_id ) {

    $output = wp_nonce_url( add_query_arg( array(
      'wpumuv_action' => 'reject',
      'user_id'       => $user_id
    ), admin_url( 'users.php?page=wpum-manage-signups' ) ), 'wpumuv_action' );

    return $output;

  }

  /**
   * Retrieve the user resend verification link.
   *
   * @since 1.0.0
   * @param  string $user_id user id number.
   * @return string
   */
  public function get_user_resend_link( $user_id ) {

    $output = wp_nonce_url( add_query_arg( array(
      'wpumuv_action' => 'resend_email',
      'user_id'       => $user_id
    ), admin_url( 'users.php?page=wpum-manage-signups' ) ), 'wpumuv_action' );

    return $output;

  }

  /**
   * Manually process the signups based on action link.
   *
   * @since 1.0.0
   * @return void
   */
  public function process_signups() {

    if( isset( $_GET['wpumuv_action'] ) && isset( $_GET['user_id'] ) ) {

      if( ! current_user_can( 'manage_options' ) )
  				return;

  		if( ! is_admin() )
  			return;

      if ( ! wp_verify_nonce( $_GET['_wpnonce'], 'wpumuv_action' ) )
        return;

      $user_id = absint( $_GET['user_id'] );
      $action  = sanitize_title( $_GET['wpumuv_action'] );

      if( $action == 'approve' ) {

        wpumuv_delete_verification_status( $user_id );
        wpumuv_send_approval_email( $user_id );

      } elseif( $action == 'reject' ) {

        $this->delete_user( $user_id );

      } elseif( $action == 'resend_email' ) {

        wpumuv_send_verification_email( $user_id );

      }

      wp_redirect( add_query_arg( array( 'message' => $action, 'wpumuv_action' => true ), admin_url( 'users.php?page=wpum-manage-signups' ) ) );
      exit;

    }

  }

  /**
   * Delete the user and notify him.
   *
   * @since 1.0.0
   * @param  string $user_id the user to delete.
   * @return void
   */
  public function delete_user( $user_id ) {

    wpumuv_send_rejection_email( $user_id );

    require_once( ABSPATH . 'wp-admin/includes/user.php' );
    wp_delete_user( $user_id );

  }

  /**
   * Show admin notices.
   *
   * @since 1.0.0
   * @return void
   */
  public function admin_notices() {

    if( isset( $_GET['message'] ) && isset( $_GET['wpumuv_action'] ) ) {

      if( $_GET['message'] == 'approve' ) {

        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php esc_html_e( 'User successfully approved.', 'wpum-user-verification' ); ?></p>
        </div>
        <?php

      }

      if( $_GET['message'] == 'reject' ) {

        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php esc_html_e( 'User successfully rejected and deleted.', 'wpum-user-verification' ); ?></p>
        </div>
        <?php

      }

      if( $_GET['message'] == 'resend_email' ) {

        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php esc_html_e( 'User successfully notified.', 'wpum-user-verification' ); ?></p>
        </div>
        <?php

      }

    }

    // Bulk messages.
    if( isset( $_POST['bulk_user'] ) ) {

      if( isset( $_POST['action'] ) ) {

        if( $_POST['action'] == 'bulk-approve' ) {

          ?>
          <div class="notice notice-success is-dismissible">
              <p><?php esc_html_e( 'Users successfully approved.', 'wpum-user-verification' ); ?></p>
          </div>
          <?php

        }

        if( $_POST['action'] == 'bulk-reject' ) {

          ?>
          <div class="notice notice-success is-dismissible">
              <p><?php esc_html_e( 'Users successfully rejected and deleted.', 'wpum-user-verification' ); ?></p>
          </div>
          <?php

        }

      }

    }

  }

  /**
   * Admin page output.
   *
   * @since 1.0.0
   * @return void
   */
  public function admin_page() {

    $pending_users = new WPUMUV_Pending_Users_Table();
		$pending_users->prepare_items();

    include 'views/html-signups-manager.php';

  }

}

global $wpumuv_signups_manager;
$wpumuv_signups_manager = new WPUMUV_Signups_Manager;
$wpumuv_signups_manager->init();
