<?php if (!defined('ABSPATH')) { exit; } // Exit if accessed directly
/**
 * Simple_Job_Board_CAPTCHA_Settings Class
 * 
 * @link        https://wordpress.org/plugins/simple-job-board
 * @since       2.3.2
 * @since       2.4.0   Improved Input/Output Sanitization & Escaping
 * 
 * @package     Simple_Job_Board_CAPTCHA_Admin
 * @subpackage  Simple_Job_Board_CAPTCHA_Admin/admin
 * @author      PressTigers <support@presstigers.com>
 */

class Simple_Job_Board_CAPTCHA_Settings {

    /**
     * Initialize the class and set its properties.
     *
     * @since   2.2.3
     */
    public function __construct() {

        // Filter -> Add Settings reCAPTCHA Tab
        add_filter('sjb_settings_tab_menus', array($this, 'add_settings_tab'), 9999);

        // Action -> Add Settings reCAPTCHA Section 
        add_action('sjb_settings_tab_section', array($this, 'add_settings_section'), 9999);

        // Action -> Save Settings reCAPTCHA Section 
        add_action('sjb_save_setting_sections', array($this, 'save_settings_section'), 9999);
    }

    /**
     * Add Settings reCAPTCHA Tab.
     *
     * @since    2.3.2
     * 
     * @param    array  $tabs  Settings Tab
     * @return   array  $tabs  Merge array of Settings Tab with reCAPTCHA Tab.
     */
    public function add_settings_tab($tabs) {
        $tabs['recaptcha'] = esc_html__('reCAPTCHA API Keys', 'simple-job-board-captcha');
        return $tabs;
    }

    /**
     * Add Settings reCAPTCHA Section.
     *
     * reCAPTCHA Options:
     * 
     * - Site Key
     * - Secret Key
     * - Language
     *  
     * @since    2.3.2
     */
    public function add_settings_section() {
        ?>
        <!-- reCAPTCHA API Keys & Languages -->
        <div id="settings-recaptcha" class="sjb-admin-settings" style="display:none;">

            <?php
            // Get reCAPTCHA Options            
            if ( get_option('job_board_recaptcha') ) {
                $recaptcha = get_option('job_board_recaptcha');
            } else {
                $recaptcha = array(
                    'site_key'      => '',
                    'secret_key'    => '',
                    'language'      => 'en',
                );
            }
            ?>
            <form method="post" id="recaptcha-form">
                <h4 class="first"><?php echo __('Manage reCAPTCHA API keys', 'simple-job-board-captcha'); ?></h4>
                <div class="sjb-section">
                    <div class="sjb-content">
                        <p><a href="https://www.google.com/recaptcha/admin" target="_blank"><?php echo __('Get the reCAPTCHA keys here', 'simple-job-board-captcha'); ?></a></p>
                        <p><strong><?php echo __('Both keys must be filled to enable reCAPTCHA on application form.', 'simple-job-board-captcha'); ?></strong></p>
                        <div class="sjb-form-group">
                            <label><?php echo __('Site Key', 'simple-job-board-captcha'); ?></label>
                            <input type="text" name="recaptcha[site_key]" value="<?php echo esc_attr( $recaptcha['site_key'] ); ?>" maxlength="50">
                        </div>
                        <div class="sjb-form-group">
                            <label><?php echo __('Secret Key', 'simple-job-board-captcha'); ?></label>
                            <input type="text" name="recaptcha[secret_key]" value="<?php echo esc_attr( $recaptcha['secret_key'] ); ?>" maxlength="50"/>
                        </div>
                        <div class="sjb-form-group">
                            <label><?php echo __('Languages', 'simple-job-board-captcha'); ?></label>
                            <?php $languages = $this->recaptcha_languages(); ?>
                            <select name="recaptcha[language]">
                                <?php
                                foreach ( $languages as $language => $language_code ) {
                                    $selected = ( trim( $language_code ) == trim( $recaptcha['language'] ) ) ? 'selected=selected' : "";
                                    echo '<option value="' . trim( $language_code ) .'" '. $selected .'>' . trim ( $language ) . '</option>';
                                }                              
                                ?>
                            </select>
                        </div>
                    </div>
                </div>
                <input type="hidden" value="1" name="admin_notices">
                <input type="submit" name="recaptcha_form_submit" id="recaptcha-form-submit" class="button button-primary" value="<?php echo __('Save Changes', 'simple-job-board'); ?>">
            </form>
        </div>
        <?php
    }

    /**
     * reCAPTCHA Languages Codes.
     *
     * @since    2.3.2
     */
    public function recaptcha_languages() {
        $recaptcha_languages = array(
            'Arabic' => 'ar',
            'Afrikaans' => 'af',
            'Amharic' => 'am',
            'Armenian' => 'hy',
            'Azerbaijani' => 'az',
            'Basque' => 'eu',
            'Bengali' => 'bn',
            'Bulgarian' => 'bg',
            'Catalan' => 'ca',
            'Chinese (Hong Kong)' => 'zh-HK',
            'Chinese (Simplified)' => 'zh-CN',
            'Chinese (Traditional)' => 'zh-TW',
            'Croatian' => 'hr',
            'Czech' => 'cs',
            'Danish' => 'da',
            'Dutch' => 'nl',
            'English (UK)' => 'en-GB',
            'English (US)' => 'en',
            'Estonian' => 'et',
            'Filipino' => 'fil',
            'Finnish' => 'fi',
            'French' => 'fr',
            'French (Canadian)' => 'fr-CA',
            'Galician' => 'gl',
            'Georgian' => 'ka',
            'German' => 'de',
            'German (Austria)' => 'de-AT',
            'German (Switzerland)' => 'de-CH',
            'Greek' => 'el',
            'Gujarati' => 'gu',
            'Hebrew' => 'iw',
            'Hindi' => 'hi',
            'Hungarain' => 'hu',
            'Icelandic' => 'is',
            'Indonesian' => 'id',
            'Italian' => 'it',
            'Japanese' => 'ja',
            'Kannada' => 'kn',
            'Korean' => 'ko',
            'Laothian' => 'lo',
            'Latvian' => 'lv',
            'Lithuanian' => 'lt',
            'Malay' => 'ms',
            'Malayalam' => 'ml',
            'Marathi' => 'mr',
            'Mongolian' => 'mn',
            'Norwegian' => 'no',
            'Persian' => 'fa',
            'Polish' => 'pl',
            'Portuguese' => 'pt',
            'Portuguese (Brazil)' => 'pt-BR',
            'Portuguese (Portugal)' => 'pt-PT',
            'Romanian' => 'ro',
            'Russian' => 'ru',
            'Serbian' => 'sr',
            'Sinhalese' => 'si',
            'Slovak' => 'sk',
            'Slovenian' => 'sl',
            'Spanish' => 'es',
            'Spanish (Latin America)' => 'es-419',
            'Swahili' => 'sw',
            'Swedish' => 'sv',
            'Tamil' => 'ta',
            'Telugu' => 'te',
            'Thai' => 'th',
            'Turkish' => 'tr',
            'Ukrainian' => 'uk',
            'Urdu' => 'ur',
            'Vietnamese' => 'vi',
            'Zulu' => 'zu',
        );

        return $recaptcha_languages;
    }

    /**
     * Save Settings reCAPTCHA Section.
     * 
     * @since    2.3.2
     */
    public function save_settings_section() {
        
        $recaptcha = filter_input(INPUT_POST, 'recaptcha', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
        
        // Update Featured Job Limit Value
        if ( !empty( $recaptcha ) ) {
            update_option( 'job_board_recaptcha', array_map( 'sanitize_text_field', $recaptcha ) );
        }
    }

}

new Simple_Job_Board_CAPTCHA_Settings();